### Roxygen-izable Documentation ----
#' Create data based on Thurstonian model equations
#' 
#' Simulates a data set of paired comparisons or ranking data based a Thurstonian
#' latent utility model.
#' 
### Inputs ----
#'
#' @param nfactor The number of factors.
#' @param nitem The number of items.
#' @param nperson The number of data points (= respondents) to simulate.
#' @param itf A vector defining the items-to-factor relation. For example
#' `c(1,1,1,2,2,2)` defines six items, the first three correspond to factor 1,
#' the second three correspond to factor 2. 
#' @param variables A vector containing the names of paired comparison variables
#' to return. If `NULL` (default), all variables are returned.
#' @param varcov A matrix defining the variance-covariance matrix of the factors.
#' @param lmu A vector defining the latent means of items. 
#' @param loadings A vector defining the loadings of items.
#' @param transitive Logical. Should the data be transitive? If `TRUE`, ranking
#' data is simulated, else paired comparison data is simulated. Defaults to `TRUE`.
#' @param var A vector containing the variances for each paired comparison. Defaults to 0.
#' @param fvalues Logical. Should simulated factor values be returned? Defaults to `FALSE`.
#' @param sim Logical. Should the simulated data be returned? Defaults to `TRUE`.
#' 
### Outputs ---- 
#' @return Returns a list containing the true factor scores and the data, or a
#' matrix containing the data.
#' 
#' 
### Examples ----
#' @examples
#' 
#' nfactor <- 4
#' nitem <- 12
#' nperson <- 1000
#' itf <- rep(1:4,3)
#' varcov <- diag(1,4)
#' 
#' # latent utility means
#' set.seed(69)
#' lmu <- runif(nitem, -1, 1)
#' loadings <- runif(nitem, 0.30, 0.95)
#' 
#' FC <- sim.data(nfactor=nfactor, nitem=nitem, nperson=nperson, itf=itf, 
#' varcov = varcov, lmu = lmu, loadings=loadings)
#'     
#' @export


### Function definition ----
sim.data <- function(nfactor,nitem,nperson,itf,variables=NULL,varcov, lmu, loadings, transitive = T, var = 0, fvalues=F, sim=T){  
  # Init parameters
  nipf <- nitem/nfactor
  npair <- nitem*(nitem-1)/2
  
  # Set loadings, means, factor scores and latent variances (of items)
  # loadings
  tmp_l <- loadings
  lambda <- matrix(0, ncol=nfactor, nrow = nitem)
  itf <- itf
  
  for(i in 1:nitem){
    lambda[i,itf[i]] <- tmp_l[i]
  }
  
  # latent means (utilities)
  lmu <- lmu
  # factor scores
  # correlated
  eta_varcov <- varcov
  # make eta
  eta <- t(MASS::mvrnorm(nperson, rep(0,nfactor), eta_varcov))
  
  if(sim==T){
    # latent variances (items)
    psi <- t(MASS::mvrnorm(nperson, rep(0,nitem), diag(1-c(tmp_l^2),nitem)))
    
    A <- designA(nitems=nitem)
    
    # simulate latent difference response
    if(transitive==FALSE){
      eps <- t(MASS::mvrnorm(nperson, rep(0,npair), diag(var,npair))) # variance pair comparison
    }else{
      eps <- t(MASS::mvrnorm(nperson, rep(0,npair), diag(.0,npair))) # variance pair comparison; 0 for rankings
    }
    
    
    pgam <- A%*%lmu #positive mean latent difference judgement eq 22 (gamma)
    At <- A%*%(lambda%*%eta+psi)
    
    lresp_r <- apply(At,2,function(x) x+pgam) # latent response ranking
    #if(compitem=='ranking'){
    #  resp <- ifelse(lresp_r<0,0,1)
    #}else{
    #  lresp <- lresp_r+eps
    #  resp <- ifelse(lresp<0,0,1)
    #}
    lresp <- lresp_r+eps
    resp <- t(ifelse(lresp<0,0,1))
    resp <- as.data.frame(resp)
    names(resp) <- i.name(1:nitem)
    
    if(is.null(variables)){
      
    }else{
      tmp <- which(i.name(1:nitem)%in%variables)
      resp <- resp[,tmp]
    }
    
  }else{
    resp <- 'You chose to not simulate the data.'
  }
  
    if(fvalues==T){
    return(list(resp,eta))
  }else{
    return(resp)
  }
  
}


